<?php
/**
 * Classe principal
 *
 * @author Hugo Ferreira da Silva
 * @link http://www.hufersil.com.br/lumine
 * @package Lumine
 */


/**
 * Define o diretorio absolute de lumine
 * @var string
 */
define('LUMINE_INCLUDE_PATH', dirname(__FILE__));

/**
 * Classe principal
 *
 * @author Hugo Ferreira da Silva
 * @link http://www.hufersil.com.br/lumine
 * @package Lumine
 */
abstract class Lumine
{
	/**
	 * Carrega arquivos do pacote
	 *
	 * @author Hugo Ferreira da Silva
	 * @return void
	 */
	public static function load()
	{
		$args = func_get_args();
		foreach($args as $libname)
		{
			$basedir = LUMINE_INCLUDE_PATH . DIRECTORY_SEPARATOR . 'lib' . DIRECTORY_SEPARATOR;
			$libname = preg_replace('@^Lumine_@', '', $libname);
			$newfile = $basedir . str_replace('_', DIRECTORY_SEPARATOR, $libname). '.php';
		
			require_once $newfile;
		}
	}
	
	/**
	 * Importa arquivos
	 *
	 * @author Hugo Ferreira da Silva
	 * @return void
	 */
	public static function import()
	{
		$args = func_get_args();
		$cn = Lumine_ConnectionManager::getInstance();
		$list = $cn->getConfigurationList();

		$cfg = array_shift( $list );
		$pacote = $cfg->getProperty('package');
		
		if( !empty($pacote) )
		{
			$pacote .= '.';
		}
		
		foreach($args as $classname)
		{
			Lumine_Util::Import( $pacote . $classname );
			
			if($cfg->getOption('auto_cast_dto')){
				$name = sprintf($cfg->getOption('dto_format'), $classname);
				self::importDTO( $name );
			}
		}
	}
	
	/**
	 * Carrega models
	 * 
	 * @author Hugo Ferreira da Silva
	 * @link http://www.hufersil.com.br
	 * @return void
	 */
	public static function loadModel(){
		$args = func_get_args();
		$cn = Lumine_ConnectionManager::getInstance();
		$list = $cn->getConfigurationList();

		$cfg = array_shift( $list );
		$path = $cfg->getProperty('class_path')
			. DIRECTORY_SEPARATOR
			. $cfg->getOption('model_path')
			. DIRECTORY_SEPARATOR;
		
		foreach($args as $classname) {
			$filename = $path . $classname . '.php';
			if(file_exists($filename)){
				require_once $filename;
			}
		}
		
	}

	/**
	 * Permite criar uma classe para uma tabela 
	 * 
	 * @author Hugo Ferreira da Silva
	 * @link http://www.hufersil.com.br
	 * @param string $tablename
	 * @return Lumine_Factory
	 */
	public static function factory($tablename){
		$list = Lumine_ConnectionManager::getInstance()->getConfigurationList();
		$cfg = array_shift($list);
		
		return $cfg->factory($tablename);
	}
	
	/**
	 * Importa o DTO relacionado a classe, quando houver
	 * @author Hugo Ferreira da Silva
	 * @link http://www.hufersil.com.br/
	 * @return void
	 */
	public static function importDTO(){
		$args = func_get_args();
		$cn = Lumine_ConnectionManager::getInstance();
		$list = $cn->getConfigurationList();

		$cfg = array_shift( $list );
		
		$path = $cfg->getProperty('class_path')
			. DIRECTORY_SEPARATOR
			. str_replace('.', DIRECTORY_SEPARATOR, $cfg->getProperty('package'))
			. DIRECTORY_SEPARATOR
			. 'dto'
			. DIRECTORY_SEPARATOR;
		
		foreach($args as $classname)
		{
			$filename = $path . $classname . $cfg->getOption('class_sufix') . '.php';
			
			if(file_exists($filename)){
				require_once $filename;
			}
		}
		
	}
	
	/**
	 * Checa se um valor e realmente nulo
	 *
	 * @param mixed $val Valor a ser comparado
	 * @author Hugo Ferreira da Silva
	 * @return boolean True se for nulo, do contratio false
	 */
	public static function is_empty($val)
	{
		return gettype($val) == 'NULL';
	}
	
	/**
	 * faz a importacao automatica de classes (autoload)
	 *
	 * @see http://br2.php.net/manual/pt_BR/function.spl-autoload-register.php
	 * @param string $clname
	 */
	public static function autoload( $clname )
	{
		$args = func_get_args();
		$cn = Lumine_ConnectionManager::getInstance();
		$list = $cn->getConfigurationList();

		
		foreach($list as $cfg)
		{
		    $sufix  = $cfg->getOption('class_sufix'); 
    		$pacote = $cfg->getProperty('package');
    		$path   = $cfg->getProperty('class_path');
    		
    		$ext    = (empty($sufix) ? '' : '.' . $sufix) . '.php';  
    		
    		if( !empty($pacote) )
    		{
    			$pacote .= '.';
    		}
    		
    		$fullpath = str_replace('.','/', $path . '/' . $pacote . $clname);
    		$fullpath .= $ext;
    		
    		if( file_exists($fullpath) )
    		{
    		    Lumine_Util::import($pacote.$clname);
    		    return;
    		}
		}
	}

	/**
	 * Destroi o objeto
	 * 
	 * @author Hugo Ferreira da Silva
	 * @link http://www.hufersil.com.br/
	 * @param Lumine_Base $obj
	 * @return void
	 */
	public static function destroy(Lumine_Base &$obj){
		$obj->destroy();
		unset($obj);
	}
}

// carrega principais dependencias
Lumine::load('Utils_Util','Utils_Crypt','Event','Events_SQLEvent','Events_FormatEvent');
Lumine::load('Exception','SQLException','EventListener','Tokenizer','Parser','Exception','Configuration','ConnectionManager','Base','Validator','Union','Factory','Model');


